<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\SubCategory;
use App\Category;
use App\Size;

class SubCategoryController extends Controller
{
    /**
     * Display a listing of the sub categories.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $subCategories = SubCategory::latest()->paginate(15);
        return view('admin.sub-categories.index', compact('subCategories'));
    }

    /**
     * Show the form for creating a new sub category.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $categories = Category::where('status', 1)->get();
        $sizes = Size::where('status', 1)->get();
        return view('admin.sub-categories.create', compact('categories', 'sizes'));
    }

    /**
     * Store a newly created sub category in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //validate form
        $validator = $request->validate([
            'name' => 'required|string|max:60|unique:sub_categories',
            'categoryName' => 'required|integer',
            'note' => 'nullable|string|max:255',
            "sizes"    => "required|array|min:1",
            "sizes.*"  => "required|string|distinct",
        ]);

        // convert sizes into string
        $sizes = implode(', ', $request->sizes);

        // store subcategory
        $subCategory = SubCategory::create([
            'name' => $request->name,
            'category_id' => $request->categoryName,
            'sizes' => $sizes,
            'note' => clean($request->note),
            'status' => $request->status
        ]);
        return redirect()->route('subCategories.index')->withSuccess('Sub category created successfully!');
    }

    /**
     * Display the specified sub category.
     *
     * @param  string  $slug
     * @return \Illuminate\Http\Response
     */
    public function show($slug)
    {
        return redirect()->route('subCategories.index');
    }

    /**
     * Show the form for editing the specified sub category.
     *
     * @param  string  $slug
     * @return \Illuminate\Http\Response
     */
    public function edit($slug)
    {
        $categories = Category::where('status', 1)->get();
        $subCategory = SubCategory::where('slug', $slug)->first();
        $selectedSizes = explode(", ", $subCategory->sizes);
        $sizes = Size::where('status', 1)->get();
        return view('admin.sub-categories.edit', compact('categories', 'subCategory', 'sizes', 'selectedSizes'));
    }

    /**
     * Update the specified sub category in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  string  $slug
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $slug)
    {
        $subCategory = SubCategory::where('slug', $slug)->first();

        // validate form
        $validator = $request->validate([
            'name' => 'required|string|max:60|unique:sub_categories,name,' . $subCategory->id,
            'categoryName' => 'required|integer',
            'note' => 'nullable|string|max:255',
            "sizes"    => "required|array|min:1",
            "sizes.*"  => "required|string|distinct",
        ]);

        // convert sizes into string
        $sizes = implode(', ', $request->sizes);

        // update subcategory
        $subCategory->update([
            'name' => $request->name,
            'category_id' => $request->categoryName,
            'sizes' => $sizes,
            'note' => clean($request->note),
            'status' => $request->status
        ]);
        return redirect()->route('subCategories.index')->withSuccess('Sub category updated successfully!');
    }

    /**
     * Remove the specified sub category from storage.
     *
     * @param  string  $slug
     * @return \Illuminate\Http\Response
     */
    public function destroy($slug)
    {
        $subCategory = SubCategory::where('slug', $slug)->first();

        // destroy sub category
        $subCategory->delete();
        return redirect()->route('subCategories.index')->withSuccess('Sub category deleted successfully!');
    }

    /**
     * Change the status of specified sub category.
     *
     * @param  string  $slug
     * @return \Illuminate\Http\Response
     */
    public function changeStatus($slug)
    {
        $subCategory = SubCategory::where('slug', $slug)->first();

        // change sub category status
        if ($subCategory->status == 1) {
            $subCategory->update([
                'status' => 0
            ]);
        } else {
            $subCategory->update([
                'status' => 1
            ]);
        }
        return redirect()->route('subCategories.index')->withSuccess('Sub category status changed successfully!');
    }
}
